// MIT © 2017 azu
"use strict";

var debug = require("debug")("negaposi-analyzer-ja");
/**
 * それぞれの単語の数
 * posi: nega = 5122 : 49983
 * 1   : 1    = 1 : 0.102474841
 * @type {{good: number, bad: number, zero: number}}
 */
var posiNegaRatio = {
    posi: 5122,
    nega: 49983,
    zero: 20
};
var defaultOptions = {
    // 辞書にない単語のスコア
    unknownWordRank: 0,
    // ポジティブな単語に対する補正値(スコアに乗算)
    positiveCorrections: 1,
    // ネガティブな単語に対する補正値(スコアに乗算)
    negativeCorrections: posiNegaRatio.posi / posiNegaRatio.nega,
    // 辞書の配列
    posiNegaDict: require("../dict/pn_ja.dic.json")
};
/**
 * @param {Object[]} tokens kuromoji.jsのtoken配列
 * @param {Object} options
 */
module.exports = function (tokens) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    if (tokens.length === 0) {
        return 0;
    }
    var score = 0;
    var unknownWordRank = options.unknownWordRank || defaultOptions.unknownWordRank;
    var posiNegaDict = options.posiNegaDict || defaultOptions.posiNegaDict;
    var negativeCorrections = options.negativeCorrections || defaultOptions.negativeCorrections;
    var positiveCorrections = options.positiveCorrections || defaultOptions.positiveCorrections;
    debug("Options: %o", {
        unknownWordRank: unknownWordRank,
        negativeCorrections: negativeCorrections,
        positiveCorrections: positiveCorrections
    });
    var scoreToken = function scoreToken(token) {
        // まずは緩く取得
        var foundDictionaries = posiNegaDict.filter(function (dict) {
            return dict["surface"] === token["surface_form"] && dict["reading"] === token["reading"] && dict["posi"] === token["posi"];
        });
        // 複数候補が出たときは厳しく判定
        var foundDict = foundDictionaries.length === 1 ? foundDictionaries[0] : foundDictionaries.find(function (dict) {
            return dict["surface"] === token["surface_form"];
        });
        if (foundDict) {
            var rank = foundDict["rank"];
            debug(token["surface_form"], rank);
            if (rank > 0) {
                return rank * positiveCorrections;
            } else if (rank < 0) {
                return rank * negativeCorrections;
            }
            return rank;
        } else {
            return unknownWordRank;
        }
    };
    tokens.forEach(function (token) {
        score += scoreToken(token);
    });
    return score / tokens.length;
};
//# sourceMappingURL=negaposi-analyzer-ja.js.map